'use strict'

entityRegistry['module']['soundShape'] = {
    extendedInfo: {
        displayName: 'Sound Shape',
        displayGroup: '3D Effects',
    },
    init: (staticConfig) => {
        return {
        }
    },
    staticConfig: [
    ],
    dynamicConfig: [
        { paramName: 'model', displayName: 'Model', type: 'model', defaultValue: ''},
        { paramName: 'initialStep', displayName: 'Initial Step', type: 'float', defaultValue: 0 },
        { paramName: 'numRepeats', displayName: 'Num Repeats', type: 'int', defaultValue: 2 },
        { paramName: 'repeatStep', displayName: 'Repeat Step', type: 'float', defaultValue: 4 },
        { paramName: 'shrinkScale', displayName: 'Shrink Scale', type: 'float', defaultValue: 0 },
        { paramName: 'brightnessScale', displayName: 'Brightness Scale', type: 'float', defaultValue: .7 },
        { paramName: 'colorSyncId', displayName: 'Color Sync Id', type: 'int', defaultValue: 2 },
        { paramName: 'colorSyncAlpha', displayName: 'Color Sync Alpha', type: 'float', defaultValue: .7 },
    ],
    actions: {
        'render': (self, frameTime, config, ctx) => {
            const {
                model,
                initialStep,
                numRepeats,
                repeatStep,
                shrinkScale,
                brightnessScale,
                colorSyncId,
                colorSyncAlpha,
            } = { ...config }

            const colorBuffer = renderer.getCurrentBuffer('color')
            const depthBuffer = renderer.getCurrentBuffer('depth')
            const brightnessBuffer = renderer.getCurrentBuffer('brightness')

            const allColors = [
                [223,239,183],
                [152,223,175],
                [95,180,156],
                [65,66,136],
                [104,45,99],
            ]
            const colors = allColors.map((color) => {
                const r = lerp(color[0], allColors[colorSyncId][0], colorSyncAlpha) / 255
                const g = lerp(color[1], allColors[colorSyncId][1], colorSyncAlpha) / 255
                const b = lerp(color[2], allColors[colorSyncId][2], colorSyncAlpha) / 255
                return [[r*.5,g*.5,b*.5], [r*.25,g*.25,b*.25], [g,r,b], [g*.75,r*.75,b*.75]]
            })
            const colorIds = []
            for (let i = 0; i < colors.length; ++i) {
                colorIds.push(renderer.getPaletteId(colors[i][0], colors[i][1]))
                renderer.getPaletteId(colors[i][2], colors[i][3])
            }

            const actualNumRepeats = Math.abs(repeatStep) > 0.01 ? numRepeats : 1

            const matScaling = m4.scaling(3, 3, 3)

            const drawRepeats = (srcVerts, vertIds, normal, colorId) => {
                let brightness = 1
                let shrink = 1

                for (let j = 0; j < actualNumRepeats; ++j) {
                    const offset = m4.multiplyVector(normal, -(initialStep + j*repeatStep))
                    const matTranslation = m4.translation(offset[0], offset[1], offset[2])
                    const matWorld = m4.multiply(matTranslation, matScaling)

                    const originalVerts = vertIds.map((v) => srcVerts[v])

                    const vMid = m4.multiplyVector(originalVerts.reduce((p, c) => {
                        return m4.addVectors(p, c)
                    }, [0, 0, 0]), 1 / vertIds.length)

                    const verts = originalVerts.map((v) => m4.addVectors(vMid, m4.multiplyVector((m4.subtractVectors(v, vMid)), shrink)))

                    const transformedVerts = renderer.transformVerts(matWorld, verts)

                    renderer.clipDrawTri_Cull_Depth(
                        colorBuffer.data, depthBuffer.data, brightnessBuffer.data,
                        transformedVerts[0], transformedVerts[1], transformedVerts[2],
                        colorId, brightness
                    )
                    renderer.clipDrawTri_Cull_Depth(
                        colorBuffer.data, depthBuffer.data, brightnessBuffer.data,
                        transformedVerts[0], transformedVerts[2], transformedVerts[1],
                        colorId, brightness
                    )

                    if (transformedVerts.length > 3) {
                        renderer.clipDrawTri_Cull_Depth(
                            colorBuffer.data, depthBuffer.data, brightnessBuffer.data,
                            transformedVerts[2], transformedVerts[3], transformedVerts[0],
                            colorId, brightness
                        )
                        renderer.clipDrawTri_Cull_Depth(
                            colorBuffer.data, depthBuffer.data, brightnessBuffer.data,
                            transformedVerts[2], transformedVerts[0], transformedVerts[3],
                            colorId, brightness
                        )
                    }

                    brightness *= brightnessScale
                    shrink *= shrinkScale
                }
            }

            for (let object = 0; object < model.objects.length; ++object) {
                const objectData = model.objects[object]
                for (let subObject = 0; subObject < objectData.subObjects.length; ++subObject) {
                    const subObjectData = objectData.subObjects[subObject]

                    for (let quad = 0; quad < subObjectData.quads.length; quad++ ) {
                        const colorId = colorIds[quad%colorIds.length]
                        drawRepeats(model.verts, subObjectData.quads[quad], model.normals[subObjectData.quadNorms[quad]], colorId)
                    }

                    for (let tri = 0; tri < subObjectData.tris.length; tri++ ) {
                        const colorId = colorIds[tri%colorIds.length]
                        drawRepeats(model.verts, subObjectData.tris[tri], model.normals[subObjectData.triNorms[tri]], colorId)
                    }
                }
            }
        }
    }
}
